"""
Cognitive Map Evaluation Utilities

This package contains modules for evaluating cognitive maps:
- extraction: Utilities for extracting and validating JSON
- graph_operations: Utilities for graph creation and comparison 
- metrics: Functions for computing evaluation metrics
- evaluation: Main evaluation logic
- io_utils: Input/output utilities for data handling and results formatting
- visualization: Utilities for visualizing cognitive maps and results
"""

# Import key functions from extraction
from .extraction import (
    extract_json_from_text,
    is_complex_format,
    is_valid_position,
    is_valid_facing,
    validate_cogmap_format,
    extract_answer,
    get_setting_from_id,
    extract_model_and_version,
    determine_answer_field
)

# Import key functions from graph_operations
from .graph_operations import (
    create_graph_from_cogmap,
    normalize_facing,
    extract_objects_with_extended_info,
    get_rotation_matrices,
    apply_rotation_to_map
)

# Import key functions from metrics
from .metrics import (
    calculate_cogmap_similarity,
    calculate_extended_cogmap_similarity
)

# Import key functions from evaluation
from .evaluation import (
    evaluate_answers_and_cogmaps,
    batch_evaluate
)

# Import key functions from io_utils
from .io_utils import (
    load_jsonl_data,
    save_json_results,
    save_csv_results,
    print_results,
    initialize_results_structure
)

# Import key functions from visualization (if it exists)
try:
    from .visualization import (
        plot_cognitive_map,
        plot_cognitive_map_comparison,
        plot_results_by_setting,
        plot_model_comparison,
        save_plot
    )
except ImportError:
    pass

__all__ = [
    # extraction
    'extract_json_from_text', 'is_complex_format', 'is_valid_position',
    'is_valid_facing', 'validate_cogmap_format', 'extract_answer', 
    'get_setting_from_id', 'extract_model_and_version', 'determine_answer_field',
    
    # graph_operations
    'create_graph_from_cogmap', 'normalize_facing', 'extract_objects_with_extended_info',
    'get_rotation_matrices', 'apply_rotation_to_map',
    
    # metrics
    'calculate_cogmap_similarity', 'calculate_extended_cogmap_similarity',
    
    # evaluation
    'evaluate_answers_and_cogmaps', 'batch_evaluate',
    
    # io_utils
    'load_jsonl_data', 'save_json_results', 'save_csv_results',
    'print_results', 'initialize_results_structure',
    
    # visualization
    'plot_cognitive_map', 'plot_cognitive_map_comparison',
    'plot_results_by_setting', 'plot_model_comparison', 'save_plot'
] 